/* src/zdusb.c
*
* Implements the functions of the ZyDAS zd1211 MAC
*
* Copyright (C) 2004 ZyDAS Inc.  All Rights Reserved.
* --------------------------------------------------------------------
*
*
*
*   The contents of this file are subject to the Mozilla Public
*   License Version 1.1 (the "License"); you may not use this file
*   except in compliance with the License. You may obtain a copy of
*   the License at http://www.mozilla.org/MPL/
*
*   Software distributed under the License is distributed on an "AS
*   IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*   implied. See the License for the specific language governing
*   rights and limitations under the License.
*
*   Alternatively, the contents of this file may be used under the
*   terms of the GNU Public License version 2 (the "GPL"), in which
*   case the provisions of the GPL are applicable instead of the
*   above.  If you wish to allow the use of your version of this file
*   only under the terms of the GPL and not to allow others to use
*   your version of this file under the MPL, indicate your decision
*   by deleting the provisions above and replace them with the notice
*   and other provisions required by the GPL.  If you do not delete
*   the provisions above, a recipient may use your version of this
*   file under either the MPL or the GPL.
*
* -------------------------------------------------------------------- */

#include <linux/version.h>

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>

#include <linux/usb.h>

#include "zd1205.h"
#include "zdusb.h"
#include "zddebug.h"
#include "zdversion.h"
#include "zd1211.h"

#define ZD1211_DBG_LEVEL    0

MODULE_AUTHOR("Yarco Yang");
MODULE_DESCRIPTION("ZyDAS 802.11b/g USB Wireless LAN adapter");
MODULE_LICENSE("GPL");

static const char driver_name[] = "zd1211";


/* table of devices that work with this driver */
static struct usb_device_id zd1211_ids [] = {
	{ USB_DEVICE(VENDOR_ZYDAS, PRODUCT_1211) },
    { USB_DEVICE(VENDOR_ZYXEL, PRODUCT_G220) },
    { USB_DEVICE(VENDOR_INEXQ, PRODUCT_0711) },
	{ }					/* Terminating entry */
};


MODULE_DEVICE_TABLE(usb, zd1211_ids);


extern struct net_device *g_dev;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
static void *zd1211_probe(struct usb_device *dev, unsigned int ifnum,
			   const struct usb_device_id *id)
#else
static int zd1211_probe(struct usb_interface *interface,
	const struct usb_device_id *id)
#endif			   
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	struct usb_interface *interface = &dev->actconfig->interface[ifnum];
#else
	struct usb_device *dev = interface_to_usbdev(interface);
#endif
	
	struct net_device *net = NULL;
	struct zd1205_private *macp = NULL;
	int vendor_id, product_id;
	int dev_index = id - zd1211_ids;
	int result = 0;
    
	//char serial_number[30];
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
    usb_get_dev(dev);
#endif    

	vendor_id = dev->descriptor.idVendor;
	product_id = dev->descriptor.idProduct;

#ifdef HMAC_DEBUG	
	printk(KERN_NOTICE "vendor_id = %04x\n", vendor_id);
	printk(KERN_NOTICE "product_id = %04x\n", product_id);

    if (dev->speed == USB_SPEED_HIGH)
        printk(KERN_NOTICE "USB 2.0 Host\n");
    else
        printk(KERN_NOTICE "USB 1.1 Host\n");  
#endif
	
	//memset(serial_number, 0, 30);
	//usb_string(dev, dev->descriptor.iSerialNumber, serial_number, 29);
	//printk("Device serial number is %s\n", serial_number);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))	
	if (usb_set_configuration(dev, dev->config[0].bConfigurationValue)) {
        printk(KERN_ERR "usb_set_configuration() failed\n");
        result = -EIO;
	 	goto fail;
	}
#endif
    
#if 1
    //kernel 2.6
	if (!(macp = kmalloc(sizeof(struct zd1205_private), GFP_KERNEL))) {
		printk(KERN_ERR "out of memory allocating device structure\n");
        result = -ENOMEM;
		goto fail;
	}
	
	memset(macp, 0, sizeof(struct zd1205_private));
#endif		


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	usb_inc_dev_use(dev);
#endif    


	net = alloc_etherdev(0);  //kernel 2.6
	//net = alloc_etherdev(sizeof (struct zd1205_private));  //kernel 2.4
	if (!net) {
		printk(KERN_ERR "zd1211: Not able to alloc etherdev struct\n");
        result = -ENOMEM;
		goto fail1;
	}
	
 	g_dev = net;  //save this for CBs use
 	//macp = net->priv; //kernel 2.4
    net->priv = macp;   //kernel 2.6
    macp->device = net;
	macp->usb = dev;
    SET_MODULE_OWNER(net);
	macp->dev_index = dev_index;
	macp->release = dev->descriptor.bcdDevice;
	printk(KERN_NOTICE "Release Ver = %04x\n", macp->release);
    macp->flags = 0;
    macp->dbg_flag = ZD1211_DBG_LEVEL;
	/* set up the endpoint information */
	/* check out the endpoints */
	macp->interface = interface;
	
	init_waitqueue_head(&macp->regSet_wait);
    init_waitqueue_head(&macp->iorwRsp_wait);
    init_waitqueue_head(&macp->term_wait);

    			
	if (!zd1211_alloc_all_urbs(macp)){
        result = -ENOMEM;
		goto fail2;
	}	

    //zd1211_DownLoadUSBCode(macp, "WS11Uext.bin", NULL, cFIRMWARE_EXT_CODE);
	if (zd1211_Download_IncludeFile(macp) != 0){
		printk(KERN_ERR "zd1211_Download_IncludeFile failed\n");
        result = -EIO;
		goto fail3;
	}

    //to enable firmware
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))    
    if (usb_set_configuration(dev, dev->config[0].bConfigurationValue)) {
#else
    if (usb_set_interface(dev, interface->altsetting[0].desc.bInterfaceNumber, 	0)){
#endif        
		printk(KERN_ERR "usb_set_configuration() failed\n");
        result = -EIO;

		goto fail3;
	}
	
    set_bit(ZD1211_RUNNING, &macp->flags);
    macp->bUSBDeveiceAttached = 1;

 	if (!zd1211_InitSetup(net, macp)){
        result = -EIO;
		goto fail3;
	}
	else{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
        usb_set_intfdata(interface, macp);
        SET_NETDEV_DEV(net, &interface->dev);
        //defer_kevent(macp, KEVENT_REGISTER_NET);
#endif

#if 1 //don't register net
        if (register_netdev(net) != 0) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))            
		    usb_set_intfdata(interface, NULL);
#endif            
            goto fail3;
        }
#endif          
	}
    
    goto done;    
	
fail3:
    zd1211_free_all_urbs(macp);	

    
fail2:
    free_netdev(net);  //kernel 2.6
    //kfree(net);
    
fail1:
    kfree(macp);
    
fail:
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
    usb_put_dev(dev);
#endif
  
    macp = NULL;

done:    
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	return macp;
#else

	return result;
#endif	
}



#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
static void zd1211_disconnect(struct usb_device *dev, void *ptr)
#else
static void zd1211_disconnect(struct usb_interface *interface)
#endif
{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	struct zd1205_private *macp = (struct zd1205_private *) usb_get_intfdata(interface);
#else
	struct zd1205_private *macp = (struct zd1205_private *)ptr;
#endif	

	if (!macp) {
		printk(KERN_ERR "unregistering non-existant device\n");
		return;
	}

    set_bit(ZD1211_UNPLUG, &macp->flags);
    macp->bUSBDeveiceAttached = 0;

    if (macp->driver_isolated){
        if (macp->device->flags & IFF_UP)
            zd1205_close(macp->device);
    }

    
	unregister_netdev(macp->device);
    
    //assuming we used keventd, it must quiesce too
    flush_scheduled_work();
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))    
	usb_dec_dev_use(dev);
#else
    usb_put_dev(interface_to_usbdev(interface));
#endif



    
	zd1211_unlink_all_urbs(macp);

	zd1211_free_all_urbs(macp);
    zd1205_clear_structs(macp->device);
    kfree(macp);
	macp = NULL;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
	usb_set_intfdata(interface, NULL);
#endif
    //ZEXIT(0);
}



static struct usb_driver zd1211_driver = {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))    
	.owner =		THIS_MODULE,
#endif    
	.name =		    driver_name,
	.probe =		zd1211_probe,
	.disconnect =	zd1211_disconnect,
	.id_table =	    zd1211_ids,
};


int __init zd1211_init(void)

{



	printk(KERN_NOTICE "%s - version %s\n",  DRIVER_NAME, VERSIONID);
	return usb_register(&zd1211_driver);
}


void __exit zd1211_exit(void)
{
	usb_deregister(&zd1211_driver);


}



module_init(zd1211_init);

module_exit(zd1211_exit);
